/*
 *  Copyright (C) 2006 Simon Funk - simonfunk@gmail.com
 *  
 *  This program is free software; you can redistribute it and/or modify it under 
 *  the terms of the GNU General Public License as published by the Free Software 
 *  Foundation; either version 2 of the License, or (at your option) any later 
 *  version.
 *  
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY 
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 *  PARTICULAR PURPOSE. See the GNU General Public License for more details.
 */

#include "Aparse.h"

// If this isn't defined, then this program will just read the
//  touch screen and print the results to stdout.
#define UseX

	// Commandline options
	static int sendButtonEvents = 1;
	static int runCalibration   = 0;
	static real xfm[6];	// 2x2 transform, x' = xfm[0] + x*xfm[1] + y*xfm[2]; y' = xfm[3] + x*xfm[4] + y*xfm[5]

#ifdef UseX
#include <X11/Xutil.h>
#include <X11/extensions/XTest.h>

	// Our connection to the X server, and caches of various values..
	static Display *xDisplay  = NULL;
	static int      xScreenNo = 0;
	static Screen  *xScreen   = NULL;
	static int      xWidth, xHeight;

	// Calibration objects..
	static GC     calGC;
	static int    calNext = 0;
	static Window calWin;
	static int    calRef[4][2];	// Where the 4 targets were placed (in screen coords)
	static int    calHit[4][2];	// Where the user clicked (in raw input coords)

static
void initX()
{
	if (xDisplay)
		return;

	if (!(xDisplay = XOpenDisplay(NULL)))
			ErFatal("Couldn't open X Display.");

	xScreenNo = DefaultScreen(xDisplay);
	xScreen   = ScreenOfDisplay(xDisplay, xScreenNo);

	xWidth  = WidthOfScreen(xScreen);
	xHeight = HeightOfScreen(xScreen);

	if (runCalibration)
		printf("Screen: %dx%dx%d\n", xWidth, xHeight, PlanesOfScreen(xScreen));
}

static
void calDrawNext()
{
	int x, y;

	XSetForeground(xDisplay, calGC, 0);
	XFillRectangle(xDisplay, calWin, calGC, 0, 0, xWidth, xHeight);

	x = calRef[calNext][0];
	y = calRef[calNext][1];
	XSetForeground(xDisplay, calGC, 0xffffff);
	XDrawLine(xDisplay, calWin, calGC, x-10, y, x+10, y);
	XDrawLine(xDisplay, calWin, calGC, x, y-10, x, y+10);
	XFlush(xDisplay);
}

static
void calibrateInit()
{
	XSetWindowAttributes attributes;
	int pt;

	initX();

	forn(pt, 4) {
		calRef[pt][0] = (pt&1)?xWidth-20:20;
		calRef[pt][1] = (pt&2)?xHeight-20:20;
	}

	calGC = XDefaultGC(xDisplay, xScreenNo);

	attributes.backing_store     = NotUseful;
	attributes.override_redirect = True;

	calWin = XCreateWindow(xDisplay,
						XDefaultRootWindow(xDisplay),	// Parent
						0, 0,							// X, Y
						xWidth, xHeight,	// Width, Height (of window)
						0, PlanesOfScreen(xScreen),	// Border width, depth of window
						InputOutput,				// Class
						CopyFromParent,				// Visual
						CWBackingStore|CWOverrideRedirect,
						&attributes);

	XMapWindow(xDisplay, calWin);

	calDrawNext();
}

static
void calibrateEvent(int x, int y, int b)
{
	static int lastb = 0;

	// Ignore anything but down-press:
	if (b == lastb)
		return;
	lastb = b;
	if (!b)
		return;

	calHit[calNext][0] = x;
	calHit[calNext][1] = y;

	calNext++;

	if (calNext == 3) {	// For now we'll just use the first three points (could average results for 4)

		real adx, ady, bdx, bdy;
		real iadx, iady, ibdx, ibdy;	// Matrix inverse of above
		real dx, dy;
		real idet;
		int i;

		// First compute the deltas in raw space:
		adx = calHit[1][0] - calHit[0][0];	// Big
		ady = calHit[1][1] - calHit[0][1];	// Small
		bdx = calHit[2][0] - calHit[0][0];	// Small
		bdy = calHit[2][1] - calHit[0][1];	// Big

		// And for the targets (we know there's no skew with these):
		dx = calRef[1][0] - calRef[0][0];
		dy = calRef[2][1] - calRef[0][1];

		/*
		printf("adx=%g\n", adx);
		printf("ady=%g\n", ady);
		printf("bdx=%g\n", bdx);
		printf("bdy=%g\n", bdy);
		printf(" dx=%g\n", dx);
		printf(" dy=%g\n", dy);
		*/

		// Now we just invert the 2x2 matrix of raw vectors
		//  and multiply by our reference vectors (dx, 0 and 0, dy)
		//  to get our scaling/skewing matrix.  That is, where m
		//  is our xfm matrix, such that (for the relative portion):
		//    x' = x*xfm[1] + y*xfm[2]; y' = x*xfm[4] + y*xfm[5]
		//
		//   [adx, ady][m1, m4] = [dx,  0]
		//   [bdx, bdy][m2, m5]   [0 , dy]
		//
		// First, invert:
		//
		//   [adx, ady]
		//   [bdx, bdy]
		//
		idet = 1. / (adx * bdy - bdx * ady);
		iadx =  bdy * idet;	// Big
		iady = -ady * idet;	// Small
		ibdx = -bdx * idet;	// Small
		ibdy =  adx * idet;	// Big

		// Then multiply:
		xfm[1] = dx * iadx;
		xfm[2] = dx * ibdx;
		xfm[4] = dy * iady;
		xfm[5] = dy * ibdy;

		// Finally, infer the shift from any reference pair (I'll use 0):
		// x' = xfm[0] + x*xfm[1] + y*xfm[2]; y' = xfm[3] + x*xfm[4] + y*xfm[5]
		xfm[0] = calRef[0][0] - calHit[0][0]*xfm[1] - calHit[0][1]*xfm[2];
		xfm[3] = calRef[0][1] - calHit[0][0]*xfm[4] - calHit[0][1]*xfm[5];

		printf("'cal=");
		forn(i, 6)
			printf("%s%g", i?",":"", xfm[i]);
		printf("'\n");

		exit(0);
	}

	calDrawNext();
}

void setPtr(int time, int xi, int yi, int b)
{
	int x, y;
	static int lastX = -1, lastY = -1, lastB = 0;
	int err;

	initX();

	x = xfm[0] + xi*xfm[1] + yi*xfm[2];
	y = xfm[3] + xi*xfm[4] + yi*xfm[5];

	if (x < 0)
		x = 0;
	else if (x >= xWidth)
		x = xWidth-1;
	if (y < 0)
		x = 0;
	else if (y >= xHeight)
		y = xHeight-1;

	if (x != lastX || y != lastY) {
		XTestFakeMotionEvent(xDisplay, xScreenNo, x, y, CurrentTime);
		lastX = x;
		lastY = y;
	}

	if (b != lastB) {
		if (sendButtonEvents)
			XTestFakeButtonEvent(xDisplay, Button1, b, CurrentTime);
		lastB = b;
	}

	XFlush(xDisplay);
}
#endif



#include <linux/joystick.h>

#if 0
#define JS_EVENT_BUTTON		0x01	/* button pressed/released */
#define JS_EVENT_AXIS		0x02	/* joystick moved */
#define JS_EVENT_INIT		0x80	/* initial state of device */

struct js_event {
	__u32 time;	/* event timestamp in milliseconds */
	__s16 value;	/* value */
	__u8 type;	/* event type */
	__u8 number;	/* axis/button number */
};
#endif

typedef struct js_event *JsEvent, JsEventS;

void helpfunc(cstring *argv)
{
	printf("This is a touchscreen driver for a fuji P1120 under kernel 2.6\n");
	printf("(But it should work fine for any device that presents as a joystick.)\n");
	printf("\n");
	printf("Run with -use to see general usage.\n");
	printf("\n");
	printf("Run with -cal to calibrate.  Click on the targets as they appear.\n");
	printf("There should be three (run it again if you only get two).\n");
	printf("\n");
	printf("Then re-run with the resulting 'cal=...' string as a parameter,\n");
	printf("and you should be up and running.  Use -nb flag for safer testing.\n");
	printf("\n");
	printf("-Simon Funk (simonfunk@gmail.com)  Apr, 2006\n");
}

Main()
{
	File fl;

	int xy[2], b, type, time;
	JsEventS evs, *ev;
	ev = &evs;

	Aparse(argv, helpfunc,
			"-nb%i<0,1,don't emulate button press events--just move cursor>", &sendButtonEvents,
			"-cal%i<1,0,calibrate>", &runCalibration,
			"cal=%r(c0,493.665)%r(c1,0.0170058)%r(c2,4.58386e-05)%r(c3,282.286)%r(c4,3.10224e-05)%r(c5,0.0101991)",
				&xfm[0], &xfm[1], &xfm[2], &xfm[3], &xfm[4], &xfm[5],
			NULL);

#ifdef UseX
	if (runCalibration)
		calibrateInit();
#endif

	xy[0] = 0;
	xy[1] = 0;
	b     = 0;
	time  = 0;

	fl = fopen_read("/dev/input/js0");

	while (fread(ev, sizeof(evs), 1, fl)) {

		// printf("%10d: val:%5d type:%3d num:%3d\n", ev->time, ev->value, ev->type, ev->number);

		type = ev->type&(JS_EVENT_BUTTON|JS_EVENT_AXIS);

		switch (type) {
		
			case JS_EVENT_BUTTON:
				if (ev->number == 0)
					b = ev->value;
#ifndef UseX
					printf("Button %d=%d\n", ev->number, ev->value);
#endif
				break;

			case JS_EVENT_AXIS:
				xy[ev->number&1] = ev->value;
				break;

			default:
				printf("Bogus event type: %d\n", ev->type);
		}
		time = ev->time;

#ifdef UseX
		if (!(ev->type & JS_EVENT_INIT)) {	// Don't propagate initialization state as true events.
			if (runCalibration)
				calibrateEvent(xy[0], xy[1], b);
			else
				setPtr(time, xy[0], xy[1], b);
		}
#else
		if (ev->type & JS_EVENT_INIT)
			printf("INIT ");

		printf(" Raw: %6d %6d  Mapped: %4d %4d  Button: %d\n",
				xy[0], xy[1],
				(int)(xfm[0] + xy[0]*xfm[1] + xy[1]*xfm[2]),
				(int)(xfm[3] + xy[0]*xfm[4] + xy[1]*xfm[5]),
				b);
#endif
	}

	fclose(fl);
}


