/*
 *  Copyright (C) 2006 Simon Funk - simonfunk@gmail.com
 *  
 *  This program is free software; you can redistribute it and/or modify it under 
 *  the terms of the GNU General Public License as published by the Free Software 
 *  Foundation; either version 2 of the License, or (at your option) any later 
 *  version.
 *  
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY 
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 *  PARTICULAR PURPOSE. See the GNU General Public License for more details.
 */
#ifndef APARSE_H
#define APARSE_H

/*
 * Does wonderful things to parse your argv
 *   list.  Everything successfully parsed
 *   will be removed from the argv array.
 * If the program is envoked with -help,
 *   the Helpfunc will be called with argv as
 *   it's one parameter, and the program will
 *   exit with status 0.
 * If the program is envoked with -use, the
 *   usage will be printed, and the program will
 *   exit with status 0.
 * If an error in the parameter list is detected,
 *   the usage will be printed, and the program
 *   will exit with a non-0 exit status.
 *
 * Use: Aparse(argv, Helpfunc,
 *                ControlString1, variables...,
 *                ControlString2, variables...,
 *                ...,
 *                NULL);
 *
 * Where:
 *    argv            is exactly as it is passed to main.
 *    Helpfunc        is a pointer to a help function, or NULL
 *                    Helpfunc will be called with argv as a parameter.
 *    ControlString    is a cstring who's format is explained below
 *    variables...    are pointers to variables corresponding to
 *                    entries in the ControlString.
 *
 * Example:
 *  Aparse(argv,NULL,
 *                "in=%f(InputFile,-)%i<0,1>",&infile,&inspecified,
 *                "out=%f(OutputFile,-)",&outfile,
 *                "-fast%i<1,0>",&fastflag,
 *                "color=%r(R,1.)%r(G,1.)%r(B,1.)",&r,&g,&b,
 *                NULL);
 *
 * Each control string matches one parameter.
 *
 * Control string format:
 *
 *    %s                means return a cstring (via a char **, NOT a char *)
 *    %c                means return a character (via an int *, NOT a char *)
 *    %f                like %s, but '-' returns NULL.  Use for Filenames.
 *    %i                means return an int (via an int *)
 *    %r                means return a real (via a real *)
 *    (name,default)    encloses a 'USE' name and default value for a
 *                      parameter.
 *     <on,off>         encloses values to assign to the matching variable
 *                      depending on whether or not the given keyword
 *                      was matched.
 *    ...               a triple dot is a special case that means it is
 *                      ok for there to be unparsed arguments left over.
 *
 * As any non-keyworded entry will match the first not-already-matched
 * arg in argv, it is a good idea to list all such entries AFTER all
 * keyworded entries.
 *
 * Additional parameter types (eg cstring, file, int, etc..) can be added
 * by making a call to AparseAddParser(key,parsefunc); before calling
 * Aparse itself.  Here, "key" is a char specifying the type indicator
 * (eg, 'i' for integer) and "parsefunc" is a function taking two parameters:
 *    int parsefunc(cstring value, pointer variable);
 * Parsefunc extracts the value from "value" leaving the result in *variable.
 * Prasefunc returns 1 on error, and 0 on success.  See AparseParseInt,
 * AparseParseReal, etc... for examples.  Also see AparseInitFuncs();
 */

#include "Toybox.h"

typedef void	(*HelpfuncPtr)(cstring* argv);
typedef int		(*ParseFunctionPtr)(cstring arg, void* val);

int 	Aparse			(cstring* argv, HelpfuncPtr help, ...);

void	AparseAddParser	(int c, ParseFunctionPtr f);


#endif
